unit TaskDialogs;

interface

{$A4}
{$Z4}
{ $DEFINE EMUONLY}

uses
  Windows, Messages;

type
  { $EXTERNALSYM PFTASKDIALOGCALLBACK}
  PFTASKDIALOGCALLBACK = function(hwnd: HWND; msg: UINT; wParam: WPARAM;
    lParam: LPARAM; lpRefData: LONG_PTR): HResult; stdcall;
  TFTaskDialogCallback = PFTASKDIALOGCALLBACK;

const
  { Task Dialog Flags }

  {$EXTERNALSYM TDF_ENABLE_HYPERLINKS}
  TDF_ENABLE_HYPERLINKS               = $0001;
  {$EXTERNALSYM TDF_USE_HICON_MAIN}
  TDF_USE_HICON_MAIN                  = $0002;
  {$EXTERNALSYM TDF_USE_HICON_FOOTER}
  TDF_USE_HICON_FOOTER                = $0004;
  {$EXTERNALSYM TDF_ALLOW_DIALOG_CANCELLATION}
  TDF_ALLOW_DIALOG_CANCELLATION       = $0008;
  {$EXTERNALSYM TDF_USE_COMMAND_LINKS}
  TDF_USE_COMMAND_LINKS               = $0010;
  {$EXTERNALSYM TDF_USE_COMMAND_LINKS_NO_ICON}
  TDF_USE_COMMAND_LINKS_NO_ICON       = $0020;
  {$EXTERNALSYM TDF_EXPAND_FOOTER_AREA}
  TDF_EXPAND_FOOTER_AREA              = $0040;
  {$EXTERNALSYM TDF_EXPANDED_BY_DEFAULT}
  TDF_EXPANDED_BY_DEFAULT             = $0080;
  {$EXTERNALSYM TDF_VERIFICATION_FLAG_CHECKED}
  TDF_VERIFICATION_FLAG_CHECKED       = $0100;
  {$EXTERNALSYM TDF_SHOW_PROGRESS_BAR}
  TDF_SHOW_PROGRESS_BAR               = $0200;
  {$EXTERNALSYM TDF_SHOW_MARQUEE_PROGRESS_BAR}
  TDF_SHOW_MARQUEE_PROGRESS_BAR       = $0400;
  {$EXTERNALSYM TDF_CALLBACK_TIMER}
  TDF_CALLBACK_TIMER                  = $0800;
  {$EXTERNALSYM TDF_POSITION_RELATIVE_TO_WINDOW}
  TDF_POSITION_RELATIVE_TO_WINDOW     = $1000;
  {$EXTERNALSYM TDF_RTL_LAYOUT}
  TDF_RTL_LAYOUT                      = $2000;
  {$EXTERNALSYM TDF_NO_DEFAULT_RADIO_BUTTON}
  TDF_NO_DEFAULT_RADIO_BUTTON         = $4000;
  {$EXTERNALSYM TDF_CAN_BE_MINIMIZED}
  TDF_CAN_BE_MINIMIZED                = $8000;

  { Task Dialog Messages }

  {$EXTERNALSYM TDM_NAVIGATE_PAGE}
  TDM_NAVIGATE_PAGE                   = WM_USER+101;
  {$EXTERNALSYM TDM_CLICK_BUTTON}
  TDM_CLICK_BUTTON                    = WM_USER+102; // wParam = Button ID
  {$EXTERNALSYM TDM_SET_MARQUEE_PROGRESS_BAR}
  TDM_SET_MARQUEE_PROGRESS_BAR        = WM_USER+103; // wParam = 0 (nonMarque) wParam != 0 (Marquee)
  {$EXTERNALSYM TDM_SET_PROGRESS_BAR_STATE}
  TDM_SET_PROGRESS_BAR_STATE          = WM_USER+104; // wParam = new progress state
  {$EXTERNALSYM TDM_SET_PROGRESS_BAR_RANGE}
  TDM_SET_PROGRESS_BAR_RANGE          = WM_USER+105; // lParam = MAKELPARAM(nMinRange, nMaxRange)
  {$EXTERNALSYM TDM_SET_PROGRESS_BAR_POS}
  TDM_SET_PROGRESS_BAR_POS            = WM_USER+106; // wParam = new position
  {$EXTERNALSYM TDM_SET_PROGRESS_BAR_MARQUEE}
  TDM_SET_PROGRESS_BAR_MARQUEE        = WM_USER+107; // wParam = 0 (stop marquee), wParam != 0 (start marquee), lparam = speed (milliseconds between repaints)
  {$EXTERNALSYM TDM_SET_ELEMENT_TEXT}
  TDM_SET_ELEMENT_TEXT                = WM_USER+108; // wParam = element (TASKDIALOG_ELEMENTS), lParam = new element text (LPCWSTR)
  {$EXTERNALSYM TDM_CLICK_RADIO_BUTTON}
  TDM_CLICK_RADIO_BUTTON              = WM_USER+110; // wParam = Radio Button ID
  {$EXTERNALSYM TDM_ENABLE_BUTTON}
  TDM_ENABLE_BUTTON                   = WM_USER+111; // lParam = 0 (disable), lParam != 0 (enable), wParam = Button ID
  {$EXTERNALSYM TDM_ENABLE_RADIO_BUTTON}
  TDM_ENABLE_RADIO_BUTTON             = WM_USER+112; // lParam = 0 (disable), lParam != 0 (enable), wParam = Radio Button ID
  {$EXTERNALSYM TDM_CLICK_VERIFICATION}
  TDM_CLICK_VERIFICATION              = WM_USER+113; // wParam = 0 (unchecked), 1 (checked), lParam = 1 (set key focus)
  {$EXTERNALSYM TDM_UPDATE_ELEMENT_TEXT}
  TDM_UPDATE_ELEMENT_TEXT             = WM_USER+114; // wParam = element (TASKDIALOG_ELEMENTS), lParam = new element text (LPCWSTR)
  {$EXTERNALSYM TDM_SET_BUTTON_ELEVATION_REQUIRED_STATE}
  TDM_SET_BUTTON_ELEVATION_REQUIRED_STATE = WM_USER+115; // wParam = Button ID, lParam = 0 (elevation not required), lParam != 0 (elevation required)
  {$EXTERNALSYM TDM_UPDATE_ICON}
  TDM_UPDATE_ICON                     = WM_USER+116; // wParam = icon element (TASKDIALOG_ICON_ELEMENTS), lParam = new icon (hIcon if TDF_USE_HICON_* was set, PCWSTR otherwise)

  { Task Dialog Notifications }

  {$EXTERNALSYM TDN_CREATED}
  TDN_CREATED                = 0;
  {$EXTERNALSYM TDN_NAVIGATED}
  TDN_NAVIGATED              = 1;
  {$EXTERNALSYM TDN_BUTTON_CLICKED}
  TDN_BUTTON_CLICKED         = 2;            // wParam = Button ID
  {$EXTERNALSYM TDN_HYPERLINK_CLICKED}
  TDN_HYPERLINK_CLICKED      = 3;            // lParam = (LPCWSTR)pszHREF
  {$EXTERNALSYM TDN_TIMER}
  TDN_TIMER                  = 4;            // wParam = Milliseconds since dialog created or timer reset
  {$EXTERNALSYM TDN_DESTROYED}
  TDN_DESTROYED              = 5;
  {$EXTERNALSYM TDN_RADIO_BUTTON_CLICKED}
  TDN_RADIO_BUTTON_CLICKED   = 6;            // wParam = Radio Button ID
  {$EXTERNALSYM TDN_DIALOG_CONSTRUCTED}
  TDN_DIALOG_CONSTRUCTED     = 7;
  {$EXTERNALSYM TDN_VERIFICATION_CLICKED}
  TDN_VERIFICATION_CLICKED   = 8;            // wParam = 1 if checkbox checked, 0 if not, lParam is unused and always 0
  {$EXTERNALSYM TDN_HELP}
  TDN_HELP                   = 9;
  {$EXTERNALSYM TDN_EXPANDO_BUTTON_CLICKED}
  TDN_EXPANDO_BUTTON_CLICKED = 10;           // wParam = 0 (dialog is now collapsed), wParam != 0 (dialog is now expanded)

type
  { $EXTERNALSYM TASKDIALOG_BUTTON}
  TASKDIALOG_BUTTON = packed record
    nButtonID: Integer;
    pszButtonText: LPCWSTR;
  end;
  { $EXTERNALSYM _TASKDIALOG_BUTTON}
  _TASKDIALOG_BUTTON = TASKDIALOG_BUTTON;
  PTaskDialogButton = ^TTaskDialogButton;
  TTaskDialogButton = TASKDIALOG_BUTTON;

const
  { Task Dialog Elements }

  {$EXTERNALSYM TDE_CONTENT}
  TDE_CONTENT              = 0;
  {$EXTERNALSYM TDE_EXPANDED_INFORMATION}
  TDE_EXPANDED_INFORMATION = 1;
  {$EXTERNALSYM TDE_FOOTER}
  TDE_FOOTER               = 2;
  {$EXTERNALSYM TDE_MAIN_INSTRUCTION}
  TDE_MAIN_INSTRUCTION     = 3;

  { Task Dialog Icon Elements }

  {$EXTERNALSYM TDIE_ICON_MAIN}
  TDIE_ICON_MAIN           = 0;
  {$EXTERNALSYM TDIE_ICON_FOOTER}
  TDIE_ICON_FOOTER         = 1;

  { Task Dialog Common Icons }

  {$EXTERNALSYM TD_WARNING_ICON}
  TD_WARNING_ICON         = MAKEINTRESOURCEW(Word(-1));
  {$EXTERNALSYM TD_ERROR_ICON}
  TD_ERROR_ICON           = MAKEINTRESOURCEW(Word(-2));
  {$EXTERNALSYM TD_INFORMATION_ICON}
  TD_INFORMATION_ICON     = MAKEINTRESOURCEW(Word(-3));
  {$EXTERNALSYM TD_SHIELD_ICON}
  TD_SHIELD_ICON          = MAKEINTRESOURCEW(Word(-4));

  { Task Dialog Button Flags }

  {$EXTERNALSYM TDCBF_OK_BUTTON}
  TDCBF_OK_BUTTON            = $0001;  // selected control return value IDOK
  {$EXTERNALSYM TDCBF_YES_BUTTON}
  TDCBF_YES_BUTTON           = $0002;  // selected control return value IDYES
  {$EXTERNALSYM TDCBF_NO_BUTTON}
  TDCBF_NO_BUTTON            = $0004;  // selected control return value IDNO
  {$EXTERNALSYM TDCBF_CANCEL_BUTTON}
  TDCBF_CANCEL_BUTTON        = $0008;  // selected control return value IDCANCEL
  {$EXTERNALSYM TDCBF_RETRY_BUTTON}
  TDCBF_RETRY_BUTTON         = $0010;  // selected control return value IDRETRY
  {$EXTERNALSYM TDCBF_CLOSE_BUTTON}
  TDCBF_CLOSE_BUTTON         = $0020;  // selected control return value IDCLOSE

type
  { $EXTERNALSYM TASKDIALOGCONFIG}
  TASKDIALOGCONFIG = packed record
    cbSize: UINT;
    hwndParent: HWND;
    hInstance: HINST;                     // used for MAKEINTRESOURCE() strings
    dwFlags: DWORD;                       // TASKDIALOG_FLAGS (TDF_XXX) flags
    dwCommonButtons: DWORD;               // TASKDIALOG_COMMON_BUTTON (TDCBF_XXX) flags
    pszWindowTitle: LPCWSTR;              // string or MAKEINTRESOURCE()
    case Integer of
      0: (hMainIcon: HICON);
      1: (pszMainIcon: LPCWSTR;
          pszMainInstruction: LPCWSTR;
          pszContent: LPCWSTR;
          cButtons: UINT;
          pButtons: PTaskDialogButton;
          nDefaultButton: Integer;
          cRadioButtons: UINT;
          pRadioButtons: PTaskDialogButton;
          nDefaultRadioButton: Integer;
          pszVerificationText: LPCWSTR;
          pszExpandedInformation: LPCWSTR;
          pszExpandedControlText: LPCWSTR;
          pszCollapsedControlText: LPCWSTR;
          case Integer of
            0: (hFooterIcon: HICON);
            1: (pszFooterIcon: LPCWSTR;
                pszFooter: LPCWSTR;
                pfCallback: TFTaskDialogCallback;
                lpCallbackData: LONG_PTR;
                cxWidth: UINT  // width of the Task Dialog's client area in DLU's.
                               // If 0, Task Dialog will calculate the ideal width.
              );
          );
  end;
  {$EXTERNALSYM _TASKDIALOGCONFIG}
  _TASKDIALOGCONFIG = TASKDIALOGCONFIG;
  PTaskDialogConfig = ^TTaskDialogConfig;
  TTaskDialogConfig = TASKDIALOGCONFIG;

{$EXTERNALSYM TaskDialogIndirect}
function TaskDialogIndirect(const pTaskConfig: TTaskDialogConfig;
  pnButton: PInteger; pnRadioButton: PInteger; pfVerificationFlagChecked: PBOOL): HRESULT;

{$EXTERNALSYM TaskDialog}
function TaskDialog(hwndParent: HWND; hInstance: HINST; pszWindowTitle,
  pszMainInstruction, pszContent: LPCWSTR; dwCommonButtons: DWORD;
  pszIcon: LPCWSTR; pnButton: PInteger): HRESULT;

implementation

uses
  Consts, Types, SysUtils, Classes, Controls, MMSystem, Graphics, Forms,
  StdCtrls, ExtCtrls, CommCtrl, ComCtrls, Buttons, ComObj;

type
  TTaskDialogEmu = class;

  TTaskForm = class(TForm)
  private
    FTaskDialog: TTaskDialogEmu;
    FRTL: Boolean;
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure TDMClickMessage(var Msg: TMessage); message TDM_CLICK_BUTTON;
    procedure TDMClickRadioButton(var Msg: TMessage); message TDM_CLICK_RADIO_BUTTON;
    procedure TDMClickVerification(var Msg: TMessage); message TDM_CLICK_VERIFICATION;
    procedure TDMEnableButton(var Msg: TMessage); message TDM_ENABLE_BUTTON;
    procedure TDMEnableRadioButton(var Msg: TMessage); message TDM_ENABLE_RADIO_BUTTON;
    procedure TDMNavigatePage(var Msg: TMessage); message TDM_NAVIGATE_PAGE;
    procedure TDMSetButtonEvevation(var Msg: TMessage); message TDM_SET_BUTTON_ELEVATION_REQUIRED_STATE;
    procedure TDMSetElementText(var Msg: TMessage); message TDM_SET_ELEMENT_TEXT;
    procedure TDMSetMarqueeProgressBar(var Msg: TMessage); message TDM_SET_MARQUEE_PROGRESS_BAR;
    procedure TDMSetProgressBarMarquee(var Msg: TMessage); message TDM_SET_PROGRESS_BAR_MARQUEE;
    procedure TDMSetProgressBarPos(var Msg: TMessage); message TDM_SET_PROGRESS_BAR_POS;
    procedure TDMSetProgressBarRange(var Msg: TMessage); message TDM_SET_PROGRESS_BAR_RANGE;
    procedure TDMSetProgressBarState(var Msg: TMessage); message TDM_SET_PROGRESS_BAR_STATE;
    procedure TDMUpdateElementText(var Msg: TMessage); message TDM_UPDATE_ELEMENT_TEXT;
    procedure TDMUpdateIcon(var Msg: TMessage); message TDM_UPDATE_ICON;
  public
    constructor CreateNew(AOwner: TComponent; Dummy: Integer  = 0); override;
    property TaskDialog: TTaskDialogEmu read FTaskDialog;
  end;

  TTaskDialogEmu = class
  private
    FForm: TForm;
    FPanelWorkArea: TPanel;
    FPanelButtons: TPanel;
    FIcon: TImage;
    FTitle: TLabel;
    FText1: TLabel;
    FText2: TLinkLabel;
    FTimer: TTimer;
    FProgressBar: TProgressBar;
    FRadioBtnFirst: TRadioButton;
    FPanelExpand: TPanel;
    FDetailedText: TLabel;
    FExpandChevron: TButton;
    FExpandLabel: TLabel;
    FCheckBox: TCheckBox;
    FBtnOK: TButton;
    FBtnYes: TButton;
    FBtnNo: TButton;
    FBtnCancel: TButton;
    FBtnRetry: TButton;
    FBtnClose: TButton;
    FBtnLast: TButton;
    FBtnDefault: TButton;
    FFooterIcon: TImage;
    FFooterText1: TLabel;
    FFooterText2: TLinkLabel;

    FTaskConfig: TTaskDialogConfig;
    FButton: PInteger;
    FRadioButton: PInteger;
    FVerificationFlagChecked: PBOOL;

    FModalResult: Integer;
    FRadioButtonResult: Integer;
    FFlagChecked: Boolean;
    FCanClose: Boolean;

    procedure Click(Sender: TObject);
    procedure Tick(Sender: TObject);
    procedure LinkClick(Sender: TObject; const Link: string; LinkType: TSysLinkType);
    procedure CloseQuery(Sender: TObject; var ACanClose: Boolean);
    procedure KeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure FormShow(Sender: TObject);
    procedure TDMClickMessage(var Msg: TMessage);
    procedure TDMClickRadioButton(var Msg: TMessage);
    procedure TDMClickVerification(var Msg: TMessage);
    procedure TDMEnableButton(var Msg: TMessage);
    procedure TDMEnableRadioButton(var Msg: TMessage);
    procedure TDMNavigatePage(var Msg: TMessage);
    procedure TDMSetButtonEvevation(var Msg: TMessage);
    procedure TDMSetElementText(var Msg: TMessage);
    procedure TDMSetMarqueeProgressBar(var Msg: TMessage);
    procedure TDMSetProgressBarMarquee(var Msg: TMessage);
    procedure TDMSetProgressBarPos(var Msg: TMessage);
    procedure TDMSetProgressBarRange(var Msg: TMessage);
    procedure TDMSetProgressBarState(var Msg: TMessage);
    procedure TDMUpdateElementText(var Msg: TMessage);
    procedure TDMUpdateIcon(var Msg: TMessage);
    function LoadText(const AText: PWideChar): String;
    function TextSize(const AFont: TFont; const AText: String; const AWidth: Integer = 0): Integer;
    procedure SetIcon(const AIcon: TIcon; const AIconH: HICON; const AIconN: PWideChar; const AUseHandle: Boolean; const AOverrideWidth: Integer = 0);
    procedure SetLabelHeight(const ALabel: TLabel); overload;
    procedure SetLabelHeight(const ALabel: TLinkLabel); overload;
  protected
    property Form: TForm read FForm;

    property TaskConfig: TTaskDialogConfig read FTaskConfig;
    property Button: PInteger read FButton;
    property RadioButton: PInteger read FRadioButton;
    property VerificationFlagChecked: PBOOL read FVerificationFlagChecked;

    property ModalResult: Integer read FModalResult;
    property RadioButtonResult: Integer read FRadioButtonResult;
    property FlagChecked: Boolean read FFlagChecked;
    property CanClose: Boolean read FCanClose;

    property PanelWorkArea: TPanel read FPanelWorkArea;
    property PanelButtons: TPanel read FPanelButtons;
    property Icon: TImage read FIcon;
    property Title: TLabel read FTitle;
    property Text1: TLabel read FText1;
    property Text2: TLinkLabel read FText2;
    property Timer: TTimer read FTimer;
    property ProgressBar: TProgressBar read FProgressBar;
    property RadioBtnFirst: TRadioButton read FRadioBtnFirst;
    property PanelExpand: TPanel read FPanelExpand;
    property DetailedText: TLabel read FDetailedText;
    property ExpandChevron: TButton read FExpandChevron;
    property ExpandLabel: TLabel read FExpandLabel;
    property CheckBox: TCheckBox read FCheckBox;
    property BtnOK: TButton read FBtnOK;
    property BtnYes: TButton read FBtnYes;
    property BtnNo: TButton read FBtnNo;
    property BtnCancel: TButton read FBtnCancel;
    property BtnRetry: TButton read FBtnRetry;
    property BtnClose: TButton read FBtnClose;
    property BtnLast: TButton read FBtnLast;
    property BtnDefault: TButton read FBtnDefault;
    property FooterIcon: TImage read FFooterIcon;
    property FooterText1: TLabel read FFooterText1;
    property FooterText2: TLinkLabel read FFooterText2;

    function SendNotify(const AMsg: Cardinal; const wParam: Cardinal = 0; const lParam: Cardinal = 0): Cardinal;

    procedure BeginUpdate;
    procedure EndUpdate;

    procedure Prepare; virtual;
    function CreateForm: HRESULT; virtual;
    function ShowModal: HRESULT; virtual;
    procedure SetResult; virtual;
  public
    constructor Create(const pTaskConfig: TTaskDialogConfig;
      pnButton: PInteger; pnRadioButton: PInteger;
      pfVerificationFlagChecked: PBOOL);
    destructor Destroy; override;
    function Execute: HRESULT;
  end;

//______________________________________________________________________________

function TaskDialogIndirectEmul(const pTaskConfig: TTaskDialogConfig;
  pnButton: PInteger; pnRadioButton: PInteger; pfVerificationFlagChecked: PBOOL): HRESULT; stdcall;
var
  TaskDialog: TTaskDialogEmu;
begin
  try
    // Arguments checks:
    if pTaskConfig.cbSize <> SizeOf(TTaskDialogConfig) then
    begin
      Result := E_INVALIDARG;
      Exit;
    end;
    if Assigned(pnButton) then
      pnButton^ := 0;
    if Assigned(pnRadioButton) then
      pnRadioButton^ := 0;
    if Assigned(pfVerificationFlagChecked) then
      pfVerificationFlagChecked^ := False;

    // Dialog call:
    TaskDialog := TTaskDialogEmu.Create(pTaskConfig, pnButton, pnRadioButton, pfVerificationFlagChecked);
    try
      Result := TaskDialog.Execute;
    finally
      FreeAndNil(TaskDialog);
    end;

  except
    on E: EOleSysError do
      Result := E.ErrorCode;
    on E: EOSError do
      Result := HResultFromWin32(E.ErrorCode);
    on E: Exception do
      Result := E_UNEXPECTED;
  end;
end;

function TaskDialogEmul(hwndParent: HWND; hInstance: HINST; pszWindowTitle,
  pszMainInstruction, pszContent: LPCWSTR; dwCommonButtons: DWORD;
  pszIcon: LPCWSTR; pnButton: PInteger): HRESULT; stdcall;
var
  TaskConfig: TTaskDialogConfig;
begin
  FillChar(TaskConfig, SizeOf(TaskConfig), 0);

  TaskConfig.cbSize := SizeOf(TaskConfig);
  TaskConfig.hwndParent := hwndParent;
  TaskConfig.hInstance := hInstance;
  TaskConfig.pszWindowTitle := pszWindowTitle;
  TaskConfig.pszMainInstruction := pszMainInstruction;
  TaskConfig.pszContent := pszContent;
  TaskConfig.dwCommonButtons := dwCommonButtons;
  TaskConfig.pszMainIcon := pszIcon;

  Result := TaskDialogIndirectEmul(TaskConfig, pnButton, nil, nil);
end;

//______________________________________________________________________________

var
  ComCtl32DLL: THandle;

procedure InitComCtl;
begin
  if ComCtl32DLL = 0 then
  begin
    ComCtl32DLL := GetModuleHandle(comctl32);
    if ComCtl32DLL = 0 then
      ComCtl32DLL := LoadLibrary(comctl32);
  end;
end;

//______________________________________________________________________________

var
  _TaskDialogIndirect: function(const pTaskConfig: TTaskDialogConfig;
    pnButton: PInteger; pnRadioButton: PInteger;
    pfVerificationFlagChecked: PBOOL): HRESULT; stdcall;

  _TaskDialog: function(hwndParent: HWND; hInstance: HINST;
    pszWindowTitle: LPCWSTR; pszMainInstruction: LPCWSTR; pszContent: LPCWSTR;
    dwCommonButtons: DWORD; pszIcon: LPCWSTR; pnButton: PInteger): HRESULT; stdcall;

function TaskDialogIndirect(const pTaskConfig: TTaskDialogConfig;
  pnButton: PInteger; pnRadioButton: PInteger; pfVerificationFlagChecked: PBOOL): HRESULT;
begin
  if not Assigned(_TaskDialogIndirect) then
  begin
    InitComCtl;
    {$IFNDEF EMUONLY}
    if ComCtl32DLL <> 0 then
      _TaskDialogIndirect := GetProcAddress(ComCtl32DLL, 'TaskDialogIndirect');
    {$ENDIF}
    if not Assigned(_TaskDialogIndirect) then
      _TaskDialogIndirect := TaskDialogIndirectEmul;
  end;
  Result := _TaskDialogIndirect(pTaskConfig, pnButton, pnRadioButton,
    pfVerificationFlagChecked);
end;

function TaskDialog(hwndParent: HWND; hInstance: HINST; pszWindowTitle,
  pszMainInstruction, pszContent: LPCWSTR; dwCommonButtons: DWORD;
  pszIcon: LPCWSTR; pnButton: PInteger): HRESULT;
begin
  if not Assigned(_TaskDialog) then
  begin
    InitComCtl;
    {$IFNDEF EMUONLY}
    if ComCtl32DLL <> 0 then
      _TaskDialog := GetProcAddress(ComCtl32DLL, 'TaskDialog');
    if not Assigned(_TaskDialog) then
      _TaskDialog := TaskDialogEmul;
    {$ENDIF}
  end;
  Result := _TaskDialog(hwndParent, hInstance, pszWindowTitle, pszMainInstruction,
    pszContent, dwCommonButtons, pszIcon, pnButton);
end;

//______________________________________________________________________________

{ TTaskDialogEmu }

constructor TTaskDialogEmu.Create(const pTaskConfig: TTaskDialogConfig;
  pnButton, pnRadioButton: PInteger; pfVerificationFlagChecked: PBOOL);
begin
  inherited Create;

  FTaskConfig := pTaskConfig;
  FButton := pnButton;
  FRadioButton := pnRadioButton;
  FVerificationFlagChecked := pfVerificationFlagChecked;
end;

destructor TTaskDialogEmu.Destroy;
begin
  FreeAndNil(FForm);
  inherited;
end;

function TTaskDialogEmu.Execute: HRESULT;
begin
  Result := CreateForm;
  if Failed(Result) then
    Exit;

  Result := ShowModal;
  if Failed(Result) then
    Exit;

  SetResult;

  Result := S_OK;
end;

procedure TTaskDialogEmu.Prepare;
begin
  FRadioButtonResult := 0;
  FModalResult := IDCANCEL;
  FFlagChecked := False;
end;

function TTaskDialogEmu.CreateForm: HRESULT;
const
  DefSpace          = 10;
  MinButtonLeft     = 45;

  procedure SetActiveControl(const AControl: TWinControl);
  begin
    if Form.ActiveControl = nil then
      Form.ActiveControl := AControl;
  end;

  procedure SetSystemUIFont(const AFont: TFont);
  var
    LogFont: TLogFont;
  begin
  if SystemParametersInfo(SPI_GETICONTITLELOGFONT, SizeOf(LogFont), @LogFont, 0) then
    AFont.Handle := CreateFontIndirect(LogFont)
  else
    AFont.Handle := GetStockObject(DEFAULT_GUI_FONT);
  end;

  procedure SetCaptionExpandDown(const ALabel: TLabel; const ACaption: String); overload;
  begin
    ALabel.AutoSize := False;
    ALabel.WordWrap := True;
    ALabel.Caption := ACaption;
    ALabel.Height := 50;
    SetLabelHeight(ALabel);
  end;

  procedure SetCaptionExpandDown(const ALabel: TLinkLabel; const ACaption: String); overload;
  begin
    ALabel.AutoSize := False;
    ALabel.Caption := ACaption;
    ALabel.Height := 50;
    SetLabelHeight(ALabel);
  end;

  procedure CreateBevel(const APanel: TPanel);
  var
    Bevel: TBevel;
  begin
    Bevel := TBevel.Create(Form);
    Bevel.Parent := APanel;
    Bevel.Shape := bsTopLine;
    Bevel.Style := bsRaised;
    Bevel.Align := alTop;
  end;

  procedure CreateExpandPanel;
  begin
    FPanelExpand := TPanel.Create(Form);
    if (TaskConfig.dwFlags and TDF_EXPAND_FOOTER_AREA) = 0 then
      PanelExpand.Parent := Form
    else
      PanelExpand.Parent := PanelButtons;
    PanelExpand.ParentBiDiMode := True;
    PanelExpand.ParentCtl3D := True;
    PanelExpand.ParentDoubleBuffered := True;
    PanelExpand.ParentShowHint := True;
    PanelExpand.ParentFont := True;
    PanelExpand.ParentColor := True;
    PanelExpand.ParentBackground := True;
    PanelExpand.Height := 20;
    PanelExpand.BevelOuter := bvNone;
    PanelExpand.BorderStyle := bsNone;
    PanelExpand.Caption := '';
    PanelExpand.Align := alBottom;
    if (TaskConfig.dwFlags and TDF_EXPAND_FOOTER_AREA) = 0 then
    begin
      PanelExpand.ParentColor := False;
      PanelExpand.ParentBackground := False;
      PanelExpand.Color := clWindow;
    end
    else
      CreateBevel(PanelExpand);
  end;

  function CreateIcon(const AParent: TPanel; var Left, Top: Integer; const AIconH: HICON; const AIconN: PWideChar; const AUseHandle: Boolean; const AOverrideWidth: Integer = 0): TImage;
  begin
    if (
         (AIconH <> 0) and
         AUseHandle
       ) or
       (
         (AIconN <> nil) and
         (not AUseHandle) and
         (
            (TaskConfig.hInstance <> 0) or
            (AIconN = TD_WARNING_ICON) or
            (AIconN = TD_ERROR_ICON) or
            (AIconN = TD_INFORMATION_ICON) or
            (AIconN = TD_SHIELD_ICON)
         )
       ) then
    begin
      Result := TImage.Create(Form);
      Result.Parent := AParent;
      Result.Width := GetSystemMetrics(SM_CXICON);
      Result.Height := GetSystemMetrics(SM_CYICON);
      Result.Left := Left;
      Result.Top := Top;
      SetIcon(Result.Picture.Icon, AIconH, AIconN, AUseHandle, AOverrideWidth);
      Result.Width := Result.Picture.Icon.Width;
      Result.Height := Result.Picture.Icon.Height;

      Left := Result.Left + Result.Width + DefSpace;
    end
    else
      Result := nil;
  end;

  function GetLocalizedCaption(const AModalResult: TModalResult): String;
  //var
  //  User32DLL: HMODULE;
  begin
    case AModalResult of
      mrOK:       Result := SOKButton;
      mrCancel:   Result := SCancelButton;
      idHelp:     Result := SHelpButton;
      mrYes:      Result := SYesButton;
      mrNo:       Result := SNoButton;
      mrClose:    Result := SCloseButton;
      mrAbort:    Result := SAbortButton;
      mrRetry:    Result := SRetryButton;
      mrIgnore:   Result := SIgnoreButton;
      mrAll:      Result := SAllButton;
    else
      Result := '';
    end;
    (*
    User32DLL := GetModuleHandle(User32);
    SetLength(Result, 10240);
    SetLength(Result, LoadString(User32DLL, 800 + AModalResult - 1, PChar(Result), Length(Result)));
    *)
  end;

  function CreateButton(var Right, Top: Integer; const AModalResult: TModalResult; const AOverrideCaption: String = ''): TButton;
  var
    Btn: TButton;
    Sz: TSize;
  begin
    Btn := TButton.Create(Form);
    Btn.Parent := PanelButtons;
    Btn.ParentBiDiMode := True;
    Btn.ParentDoubleBuffered := True;
    Btn.ParentShowHint := True;
    Btn.ParentFont := True;
    Btn.WordWrap := True;
    Btn.Width := 75;
    Btn.Height := 25;
    Btn.OnClick := Click;
    Btn.ModalResult := AModalResult;
    Btn.Default := (TaskConfig.nDefaultButton = AModalResult);
    Btn.Cancel := (AModalResult = mrCancel);
    FCanClose := Btn.Cancel or FCanClose;

    if AOverrideCaption = '' then
      Btn.Caption := GetLocalizedCaption(AModalResult)
    else
      Btn.Caption := AOverrideCaption;

    FillChar(Sz, SizeOf(Sz), 0);
    if Button_GetIdealSize(Btn.Handle, Sz) then
    begin
      Btn.Tag := Sz.cx + DefSpace * 2;
      if Btn.Width < Btn.Tag then
        Btn.Width := Btn.Tag;
      Btn.Tag := Sz.cy;
      if Btn.Height < Btn.Tag then
        Btn.Height := Btn.Tag;
    end;

    Right := Right - DefSpace - Btn.Width;
    Btn.Left := Right;
    Btn.Top := Top;

    FBtnLast := Btn;
    if Btn.Default then
      FBtnDefault := Btn;
    Result := Btn;
  end;

  function CreateCommandLink(var Left, Top: Integer; const AModalResult: TModalResult; const AOverrideCaption: String = ''): TButton;
  var
    Btn: TButton;
    Sz: TSize;
    X: Integer;
  begin
    Btn := TButton.Create(Form);
    Btn.Parent := PanelWorkArea;
    Btn.ParentBiDiMode := True;
    Btn.ParentDoubleBuffered := True;
    Btn.ParentShowHint := True;
    Btn.ParentFont := True;
    Btn.WordWrap := True;
    Btn.Left := Left;
    Btn.Top := Top;
    Btn.Width := PanelWorkArea.ClientWidth - Btn.Left - DefSpace;
    Btn.Height := 25;
    Btn.OnClick := Click;
    Btn.ModalResult := AModalResult;
    Btn.Default := (TaskConfig.nDefaultButton = AModalResult);
    Btn.Cancel := (AModalResult = mrCancel);
    FCanClose := Btn.Cancel or FCanClose;

    if Win32MajorVersion >= 6 then
    begin
      if AOverrideCaption = '' then
        Btn.Caption := GetLocalizedCaption(AModalResult)
      else
        Btn.Caption := AOverrideCaption;

      X := Pos(#10, Btn.Caption);
      if X > 0 then
      begin
        Btn.CommandLinkHint := Trim(Copy(Btn.Caption, X + 1, MaxInt));
        Btn.Caption := Trim(Copy(Btn.Caption, 1, X - 1));
      end;

      Btn.Style := bsCommandLink;
      Btn.Height := Btn.Height + DefSpace;
    end
    else
    begin
      if AOverrideCaption = '' then
        Btn.Caption := GetLocalizedCaption(AModalResult)
      else
        Btn.Caption := AOverrideCaption;

      FillChar(Sz, SizeOf(Sz), 0);
      Sz.cx := Btn.Width;
      if Button_GetIdealSize(Btn.Handle, Sz) then
      begin
        Btn.Tag := Sz.cy;
        if Btn.Height < Btn.Tag then
          Btn.Height := Btn.Tag;
      end;
    end;
    Top := Btn.Top + Btn.Height;

    FBtnLast := Btn;
    if Btn.Default then
      FBtnDefault := Btn;
    Result := Btn;
  end;

  procedure CreateButtons(var Right, Top: Integer);
  var
    X: Integer;
    Btn: PTaskDialogButton;
    CustomButtons: array of PTaskDialogButton;
  begin
    if TaskConfig.dwCommonButtons <> 0 then
    begin
      if (TaskConfig.dwCommonButtons and TDCBF_CLOSE_BUTTON) <> 0 then
        FBtnClose := CreateButton(Right, Top, mrClose);
      if (TaskConfig.dwCommonButtons and TDCBF_CANCEL_BUTTON) <> 0 then
        FBtnCancel := CreateButton(Right, Top, mrCancel);
      if (TaskConfig.dwCommonButtons and TDCBF_RETRY_BUTTON) <> 0 then
        FBtnRetry := CreateButton(Right, Top, mrRetry);
      if (TaskConfig.dwCommonButtons and TDCBF_NO_BUTTON) <> 0 then
        FBtnNo := CreateButton(Right, Top, mrNo);
      if (TaskConfig.dwCommonButtons and TDCBF_YES_BUTTON) <> 0 then
        FBtnYes := CreateButton(Right, Top, mrYes);
      if (TaskConfig.dwCommonButtons and TDCBF_OK_BUTTON) <> 0 then
        FBtnOK := CreateButton(Right, Top, mrOk);
    end;

    if (TaskConfig.dwFlags and (TDF_USE_COMMAND_LINKS or TDF_USE_COMMAND_LINKS_NO_ICON)) <> 0 then
      Exit;

    Btn := TaskConfig.pButtons;
    SetLength(CustomButtons, TaskConfig.cButtons);
    for X := 0 to TaskConfig.cButtons - 1 do
    begin
      CustomButtons[X] := Btn;
      Inc(Btn);
    end;

    for X := High(CustomButtons) downto 0 do
    begin
      Btn := CustomButtons[X];
      CreateButton(Right, Top, Btn^.nButtonID, LoadText(Btn^.pszButtonText));
    end;
  end;

  procedure CreateRadioButtons(var Left, Top: Integer);

    function CreateRadioButton(var Left, Top: Integer; const AModalResult: Integer; const AText: String): TRadioButton;
    begin
      Result := TRadioButton.Create(Form);
      Result.Parent := PanelWorkArea;
      Result.ParentBiDiMode := True;
      Result.ParentColor := True;
      Result.ParentCtl3D := True;
      Result.ParentDoubleBuffered := True;
      Result.ParentFont := True;
      Result.ParentShowHint := True;
      Result.ParentCustomHint := True;
      Result.Top := Top;
      Result.Left := Left;
      Result.Width := PanelWorkArea.ClientWidth - Result.Left - DefSpace;
      Result.WordWrap := False;
      Result.Tag := AModalResult;
      Result.Caption := AText;
      Result.OnClick := Click;
      if AModalResult = TaskConfig.nDefaultRadioButton then
      begin
        FRadioBtnFirst := Result;
        Result.Checked := True;
      end;

      Top := Result.Top + Result.Height + DefSpace;
      if RadioBtnFirst = nil then
        FRadioBtnFirst := Result;
    end;

  var
    X: Integer;
    Btn: PTaskDialogButton;
  begin
    if TaskConfig.cRadioButtons <= 0 then
      Exit;

    Btn := TaskConfig.pRadioButtons;
    for X := 0 to TaskConfig.cRadioButtons - 1 do
    begin
      CreateRadioButton(Left, Top, Btn^.nButtonID, LoadText(Btn^.pszButtonText));

      Inc(Btn);
    end;

    Top := Top + DefSpace;
  end;

  procedure CreateCommandLinks(var Left, Top: Integer);
  var
    X: Integer;
    Btn: PTaskDialogButton;
  begin
    if (TaskConfig.dwFlags and (TDF_USE_COMMAND_LINKS or TDF_USE_COMMAND_LINKS_NO_ICON)) = 0 then
      Exit;

    Btn := TaskConfig.pButtons;
    for X := 0 to TaskConfig.cButtons - 1 do
    begin
      CreateCommandLink(Left, Top, Btn^.nButtonID, LoadText(Btn^.pszButtonText));

      Inc(Btn);
    end;

    if TaskConfig.cButtons > 0 then
      Top := Top + DefSpace;
  end;

var
  Left: Integer;
  Top: Integer;
  X, MinLeft: Integer;
begin
  FForm := TTaskForm.CreateNew(nil, Integer(Self));
  if TaskConfig.hwndParent <> 0 then
    Form.RecreateAsPopup(TaskConfig.hwndParent);
  Form.PopupMode := pmExplicit;
  SetSystemUIFont(Form.Font);
  Form.BorderStyle := bsDialog;
  Form.AutoSize := False;
  Form.AutoScroll := False;
  if (TaskConfig.dwFlags and TDF_CAN_BE_MINIMIZED) <> 0 then
    Form.BorderIcons := [biSystemMenu, biMinimize]
  else
    Form.BorderIcons := [biSystemMenu];
  if TaskConfig.cxWidth <> 0 then
    Form.ClientWidth := TaskConfig.cxWidth
  else
    Form.ClientWidth := 454;
  Form.ClientHeight := 80;
  if TaskConfig.pszWindowTitle = nil then
    Form.Caption := ExtractFileName(GetModuleName(0))
  else
    Form.Caption := LoadText(TaskConfig.pszWindowTitle);
  if (TaskConfig.dwFlags and TDF_POSITION_RELATIVE_TO_WINDOW) <> 0 then
    Form.Position := poOwnerFormCenter
  else
    Form.Position := poScreenCenter;
  FCanClose := (TaskConfig.dwFlags and TDF_ALLOW_DIALOG_CANCELLATION) <> 0;
  Form.OnCloseQuery := CloseQuery;
  Form.KeyPreview := True;
  Form.OnKeyDown := KeyDown;
  Form.OnShow := FormShow;

  if (TaskConfig.dwFlags and TDF_EXPAND_FOOTER_AREA) = 0 then
    CreateExpandPanel;

  FPanelButtons := TPanel.Create(Form);
  PanelButtons.Parent := Form;
  PanelButtons.ParentBiDiMode := True;
  PanelButtons.ParentCtl3D := True;
  PanelButtons.ParentDoubleBuffered := True;
  PanelButtons.ParentShowHint := True;
  PanelButtons.ParentFont := True;
  PanelButtons.ParentColor := True;
  PanelButtons.ParentBackground := True;
  PanelButtons.Height := 46;
  PanelButtons.BevelOuter := bvNone;
  PanelButtons.BorderStyle := bsNone;
  PanelButtons.Caption := '';
  PanelButtons.Align := alBottom;
  CreateBevel(PanelButtons);

  if (TaskConfig.dwFlags and TDF_EXPAND_FOOTER_AREA) <> 0 then
    CreateExpandPanel;

  FPanelWorkArea := TPanel.Create(Form);
  PanelWorkArea.Parent := Form;
  PanelWorkArea.ParentBiDiMode := True;
  PanelWorkArea.ParentCtl3D := True;
  PanelWorkArea.ParentDoubleBuffered := True;
  PanelWorkArea.ParentShowHint := True;
  PanelWorkArea.ParentFont := True;
  PanelWorkArea.ParentColor := False;
  PanelWorkArea.ParentBackground := False;
  PanelWorkArea.Color := clWindow;
  PanelWorkArea.BorderStyle := bsNone;
  PanelWorkArea.BevelOuter := bvNone;
  PanelWorkArea.Caption := '';
  PanelWorkArea.Align := alClient;

  if Assigned(PanelExpand) then
    PanelExpand.Hide;

// ____Construct buttons panel_____

  Left := PanelWorkArea.ClientWidth;
  Top := DefSpace;

  if (TaskConfig.dwCommonButtons = 0) and
     (TaskConfig.cButtons = 0) then
    FTaskConfig.dwCommonButtons := TaskConfig.dwCommonButtons or TDCBF_OK_BUTTON;

  CreateButtons(Left, Top);

  if BtnLast <> nil then
  begin
    if BtnDefault = nil then
    begin
      FBtnDefault := BtnLast;
      BtnLast.Default := True;
    end;
    if (BtnLast is TButton) and (TButton(BtnLast).Style = bsPushButton) then
      Top := BtnLast.Top + BtnLast.Height + DefSpace;
  end;

  MinLeft := MinButtonLeft;
  if TaskConfig.pszExpandedInformation <> nil then
  begin
    if TaskConfig.pszExpandedControlText = nil then
      FTaskConfig.pszExpandedControlText := TaskConfig.pszCollapsedControlText;
    if TaskConfig.pszCollapsedControlText = nil then
      FTaskConfig.pszCollapsedControlText := TaskConfig.pszExpandedControlText;

    FExpandChevron := TButton.Create(Form);
    ExpandChevron.Parent := PanelButtons;
    ExpandChevron.ParentBiDiMode := True;
    ExpandChevron.ParentDoubleBuffered := True;
    ExpandChevron.ParentFont := True;
    ExpandChevron.ParentShowHint := True;
    ExpandChevron.ParentCustomHint := True;
    ExpandChevron.Caption := '\/'; // Do Not Localize
    ExpandChevron.Left := DefSpace;
    ExpandChevron.Top := DefSpace;
    ExpandChevron.Width := ExpandChevron.Height;
    ExpandChevron.OnClick := Click;

    Top := ExpandChevron.Top + ExpandChevron.Height + DefSpace;
    if MinLeft < ExpandChevron.Left + ExpandChevron.Width then
      MinLeft := ExpandChevron.Left + ExpandChevron.Width;

    if TaskConfig.pszCollapsedControlText <> nil then
    begin
      FExpandLabel := TLabel.Create(Form);
      ExpandLabel.Parent := PanelButtons;
      ExpandLabel.ParentBiDiMode := True;
      ExpandLabel.ParentColor := True;
      ExpandLabel.ParentFont := True;
      ExpandLabel.ParentShowHint := True;
      ExpandLabel.ParentCustomHint := True;
      ExpandLabel.ShowAccelChar := False;
      ExpandLabel.Caption := LoadText(TaskConfig.pszCollapsedControlText);
      ExpandLabel.Left := ExpandChevron.Left + ExpandChevron.Width + DefSpace;
      ExpandLabel.Top := DefSpace;
      ExpandLabel.Width := TextSize(ExpandLabel.Font, ExpandLabel.Caption);
      ExpandLabel.Tag := TextSize(ExpandLabel.Font, LoadText(TaskConfig.pszExpandedControlText));
      if ExpandLabel.Width < ExpandLabel.Tag then
        ExpandLabel.Width := ExpandLabel.Tag;
      ExpandLabel.Height := TextSize(ExpandLabel.Font, ExpandLabel.Caption, ExpandLabel.Width);
      ExpandLabel.Tag := TextSize(ExpandLabel.Font, LoadText(TaskConfig.pszExpandedControlText), ExpandLabel.Width);
      if ExpandLabel.Height < ExpandLabel.Tag then
        ExpandLabel.Height := ExpandLabel.Tag;

      ExpandLabel.Tag := ExpandLabel.Top + ExpandLabel.Height + DefSpace;
      if Top < ExpandLabel.Tag then
        Top := ExpandLabel.Tag;
      if MinLeft < ExpandLabel.Left + ExpandLabel.Width then
        MinLeft := ExpandLabel.Left + ExpandLabel.Width;
    end;

    FDetailedText := TLabel.Create(Form);
    DetailedText.Parent := PanelExpand;
    DetailedText.ParentBiDiMode := True;
    DetailedText.ParentColor := True;
    DetailedText.ParentFont := True;
    DetailedText.ParentShowHint := True;
    DetailedText.ParentCustomHint := True;
    DetailedText.AutoSize := False;
    DetailedText.ShowAccelChar := False;
    DetailedText.WordWrap := True;
    DetailedText.Caption := LoadText(TaskConfig.pszExpandedInformation);
    DetailedText.Top := DefSpace;
  end;

  if TaskConfig.pszVerificationText <> nil then
  begin
    FCheckBox := TCheckBox.Create(Form);
    CheckBox.Parent := PanelButtons;
    CheckBox.ParentBiDiMode := True;
    CheckBox.ParentColor := True;
    CheckBox.ParentCtl3D := True;
    CheckBox.ParentDoubleBuffered := True;
    CheckBox.ParentFont := True;
    CheckBox.ParentShowHint := True;
    CheckBox.ParentCustomHint := True;
    CheckBox.Checked := (TaskConfig.dwFlags and TDF_VERIFICATION_FLAG_CHECKED) <> 0;
    CheckBox.Caption := LoadText(TaskConfig.pszVerificationText);
    CheckBox.Left := DefSpace;
    if ExpandChevron = nil then
    begin
      if Top <> DefSpace then
        CheckBox.Height := Top - DefSpace * 2;
      Top := DefSpace;
    end;
    CheckBox.Top := Top;
    CheckBox.OnClick := Click;
    CheckBox.Width := TextSize(CheckBox.Font, CheckBox.Caption) + CheckBox.Height + DefSpace;
    if MinLeft < CheckBox.Left + CheckBox.Width then
      MinLeft := CheckBox.Left + CheckBox.Width;
    Top := CheckBox.Top + CheckBox.Height + DefSpace;
  end;

  X := DefSpace;
  FFooterIcon := CreateIcon(PanelButtons, X, Top, TaskConfig.hFooterIcon, TaskConfig.pszFooterIcon, (TaskConfig.dwFlags and TDF_USE_HICON_FOOTER) <> 0, 24);

  if TaskConfig.pszFooter <> nil then
  begin
    if (TaskConfig.dwFlags and TDF_ENABLE_HYPERLINKS) = 0 then
    begin
      FFooterText1 := TLabel.Create(Form);
      FooterText1.Parent := PanelButtons;
      FooterText1.ParentBiDiMode := True;
      FooterText1.ParentShowHint := True;
      FooterText1.ParentFont := True;
      FooterText1.ParentColor := True;
      FooterText1.AutoSize := False;
      FooterText1.Left := X;
      FooterText1.Top := Top;
      FooterText1.Width := PanelWorkArea.ClientWidth - FooterText1.Left - DefSpace;
      FooterText1.ShowAccelChar := False;
      SetCaptionExpandDown(FooterText1, LoadText(TaskConfig.pszFooter));
      Top := FooterText1.Top + FooterText1.Height + DefSpace;
    end
    else
    begin
      FFooterText2 := TLinkLabel.Create(Form);
      FooterText2.Parent := PanelButtons;
      FooterText2.ParentShowHint := True;
      FooterText2.ParentFont := True;
      FooterText2.ParentColor := True;
      FooterText2.AutoSize := False;
      FooterText2.OnLinkClick := LinkClick;
      FooterText2.Left := X;
      FooterText2.Top := Top;
      FooterText2.Width := PanelWorkArea.ClientWidth - FooterText2.Left - DefSpace;
      SetCaptionExpandDown(FooterText2, LoadText(TaskConfig.pszFooter));
      Top := FooterText2.Top + FooterText2.Height + DefSpace;
    end;
  end;

  // Correct form height
  if Top <> DefSpace then
    PanelButtons.Height := Top;
  if (PanelButtons.ControlCount = 1) or // bevel only
     (
       (PanelExpand.Parent = PanelButtons) and
       (PanelExpand.ControlCount = 0) and
       (PanelButtons.ControlCount = 2)
     ) then
    PanelButtons.Hide;

  // Correct form width
  if Left < MinLeft then
  begin
    Top := (MinLeft - Left);
    Form.ClientWidth := Form.ClientWidth + Top;
    for X := 0 to PanelButtons.ControlCount - 1 do
      if (PanelButtons.Controls[X] is TButton) and (PanelButtons.Controls[X] <> ExpandChevron) then
        PanelButtons.Controls[X].Left := PanelButtons.Controls[X].Left + Top;
  end;

// ____Construct working area panel_____

  Left := DefSpace;
  Top := DefSpace;

  FIcon := CreateIcon(PanelWorkArea, Left, Top, TaskConfig.hMainIcon, TaskConfig.pszMainIcon, (TaskConfig.dwFlags and TDF_USE_HICON_MAIN) <> 0);

  if TaskConfig.pszMainInstruction <> nil then
  begin
    FTitle := TLabel.Create(Form);
    Title.Parent := PanelWorkArea;
    Title.ParentBiDiMode := True;
    Title.ParentShowHint := True;
    Title.ParentFont := True;
    Title.ParentColor := True;
    Title.AutoSize := False;
    Title.Left := Left;
    Title.Top := Top;
    Title.Width := PanelWorkArea.ClientWidth - Title.Left - DefSpace;
    Title.ShowAccelChar := False;
    Title.Font.Size := Round(Title.Font.Size * 1.3);
    Title.Font.Color := $CE5A00;
    SetCaptionExpandDown(Title, LoadText(TaskConfig.pszMainInstruction));
    Top := Title.Top + Title.Height + DefSpace;
  end;

  if TaskConfig.pszContent <> nil then
  begin
    Top := Top + DefSpace div 2;
    if (TaskConfig.dwFlags and TDF_ENABLE_HYPERLINKS) = 0 then
    begin
      FText1 := TLabel.Create(Form);
      Text1.Parent := PanelWorkArea;
      Text1.ParentBiDiMode := True;
      Text1.ParentShowHint := True;
      Text1.ParentFont := True;
      Text1.ParentColor := True;
      Text1.AutoSize := False;
      Text1.Left := Left;
      Text1.Top := Top;
      Text1.Width := PanelWorkArea.ClientWidth - Text1.Left - DefSpace;
      Text1.ShowAccelChar := False;
      SetCaptionExpandDown(Text1, LoadText(TaskConfig.pszContent));
      Top := Text1.Top + Text1.Height + DefSpace;
    end
    else
    begin
      FText2 := TLinkLabel.Create(Form);
      Text2.Parent := PanelWorkArea;
      Text2.ParentShowHint := True;
      Text2.ParentFont := True;
      Text2.ParentColor := True;
      Text2.AutoSize := False;
      Text2.OnLinkClick := LinkClick;
      Text2.Left := Left;
      Text2.Top := Top;
      Text2.Width := PanelWorkArea.ClientWidth - Text2.Left - DefSpace;
      SetCaptionExpandDown(Text2, LoadText(TaskConfig.pszContent));
      Top := Text2.Top + Text2.Height + DefSpace;
    end;
  end;

  if ((TaskConfig.dwFlags and TDF_SHOW_PROGRESS_BAR) <> 0) or
     ((TaskConfig.dwFlags and TDF_SHOW_MARQUEE_PROGRESS_BAR) <> 0) then
  begin
    FProgressBar := TProgressBar.Create(Form);
    ProgressBar.Parent := PanelWorkArea;
    ProgressBar.Top := Top;
    ProgressBar.Left := Left;
    ProgressBar.Width := Form.ClientWidth - ProgressBar.Left - DefSpace;
    if (TaskConfig.dwFlags and TDF_SHOW_MARQUEE_PROGRESS_BAR) = 0 then
      ProgressBar.Style := pbstNormal
    else
      ProgressBar.Style := pbstMarquee;
    ProgressBar.ParentDoubleBuffered := True;
    ProgressBar.ParentShowHint := True;
    ProgressBar.ParentCustomHint := True;
    ProgressBar.Smooth := True;
    ProgressBar.SmoothReverse := True;
    Top := ProgressBar.Top + ProgressBar.Height + DefSpace;
  end;

  CreateRadioButtons(Left, Top);
  if Assigned(RadioBtnFirst) and
     (not RadioBtnFirst.Checked) and
     ((TaskConfig.dwFlags and TDF_NO_DEFAULT_RADIO_BUTTON) = 0) then
    RadioBtnFirst.Checked := True;

  CreateCommandLinks(Left, Top);

  // Corrent form height
  if Assigned(Icon) and (Top < Icon.Top + Icon.Height) then
    Top := Icon.Top + Icon.Height;

  if Assigned(DetailedText) then
  begin
    if (TaskConfig.dwFlags and TDF_EXPAND_FOOTER_AREA) = 0 then
    begin
      DetailedText.Left := Left;
      DetailedText.Top := 0;
    end
    else
    begin
      DetailedText.Left := DefSpace;
      DetailedText.Top := DefSpace;
    end;
    DetailedText.Width := Form.ClientWidth - DetailedText.Left - DefSpace;
    DetailedText.Height := TextSize(DetailedText.Font, DetailedText.Caption, DetailedText.Width);

    PanelExpand.Height := DetailedText.Top + DetailedText.Height + DefSpace;
  end;

  Top := (Top + DefSpace div 2);
  if PanelButtons.Visible then
    Top := Top + PanelButtons.Height;
  if Form.ClientHeight < Top then
    Form.ClientHeight := Top;

// ____Finishing move_____

  SetActiveControl(BtnDefault);
  if CanClose then
    Form.BorderIcons := [biSystemMenu]
  else
    Form.BorderIcons := [];

  if (TaskConfig.dwFlags and TDF_EXPANDED_BY_DEFAULT) <> 0 then
    ExpandChevron.Click;

  SendNotify(TDN_DIALOG_CONSTRUCTED);

  if (TaskConfig.dwFlags and TDF_CALLBACK_TIMER) <> 0 then
  begin
    FTimer := TTimer.Create(Form);
    Timer.Interval := 200;
    Timer.OnTimer := Tick;
    Timer.Tag := GetTickCount;
  end;

  Result := S_OK;
end;

function TTaskDialogEmu.ShowModal: HRESULT;
begin
  SendNotify(TDN_CREATED);

  if Assigned(Timer) then
  begin
    Timer.Tag := GetTickCount;
    Timer.Enabled := True;
  end;
  FModalResult := FForm.ShowModal;

  SendNotify(TDN_DESTROYED);

  Result := S_OK;
end;

procedure TTaskDialogEmu.SetResult;
begin
  if Assigned(Button) then
    Button^ := ModalResult;
  if Assigned(RadioButton) then
    RadioButton^ := RadioButtonResult;
  if Assigned(VerificationFlagChecked) then
    VerificationFlagChecked^ := FlagChecked;
end;

procedure TTaskDialogEmu.FormShow(Sender: TObject);
const
  SND_SYSTEM = $00200000;
var
  Flags: Cardinal;
  ID: Integer;
begin
  ID := 0;
  if (TaskConfig.dwFlags and TDF_USE_HICON_MAIN) = 0 then
  begin
    if TaskConfig.pszMainIcon = TD_WARNING_ICON then
      ID := SND_ALIAS_SYSTEMEXCLAMATION
    else
    if TaskConfig.pszMainIcon = TD_ERROR_ICON then
      ID := SND_ALIAS_SYSTEMHAND
    else
    if TaskConfig.pszMainIcon = TD_INFORMATION_ICON then
      ID := SND_ALIAS_SYSTEMASTERISK;
  end;

  if ID <> 0 then
  begin
    Flags := SND_NODEFAULT or SND_ASYNC or SND_ALIAS_ID;
    if Win32MajorVersion >= 6 then
      Flags := Flags or SND_SYSTEM;

    PlaySound(PChar(ID), 0, Flags);
  end;
end;

procedure TTaskDialogEmu.BeginUpdate;
begin
  SendMessage(Form.Handle, WM_SETREDRAW, 0, 0);
end;

procedure TTaskDialogEmu.EndUpdate;
begin
  SendMessage(Form.Handle, WM_SETREDRAW, 1, 0);
  RedrawWindow(Form.Handle, nil, 0, RDW_ERASE or RDW_FRAME or RDW_INVALIDATE or RDW_ALLCHILDREN);
end;

procedure TTaskDialogEmu.Click(Sender: TObject);
begin
  if Assigned(ExpandChevron) and (Sender = ExpandChevron) then
  begin
    BeginUpdate;
    try
      if ExpandChevron.Tag = 0 then
      begin
        ExpandChevron.Tag := 1;
        ExpandChevron.Caption := '/\'; // Do Not Localize
        Form.Height := Form.Height + PanelExpand.Height;
        if PanelExpand.Parent = PanelButtons then
          PanelButtons.Height := PanelButtons.Height + PanelExpand.Height;
        PanelExpand.Show;
        if Assigned(ExpandLabel) then
          ExpandLabel.Caption := LoadText(TaskConfig.pszExpandedControlText);
      end
      else
      begin
        ExpandChevron.Tag := 0;
        ExpandChevron.Caption := '\/'; // Do Not Localize
        PanelExpand.Hide;
        if PanelExpand.Parent = PanelButtons then
          PanelButtons.Height := PanelButtons.Height - PanelExpand.Height;
        Form.Height := Form.Height - PanelExpand.Height;
        if Assigned(ExpandLabel) then
          ExpandLabel.Caption := LoadText(TaskConfig.pszCollapsedControlText);
      end;
    finally
      EndUpdate;
    end;
    SendNotify(TDN_EXPANDO_BUTTON_CLICKED, ExpandChevron.Tag, 0);
  end
  else
  if Sender is TCheckBox then
  begin
    SendNotify(TDN_VERIFICATION_CLICKED, Ord(TCheckBox(Sender).Checked), 0);
    FFlagChecked := TCheckBox(Sender).Checked;
  end
  else
  if Sender is TRadioButton then
  begin
    SendNotify(TDN_RADIO_BUTTON_CLICKED, (Sender as TRadioButton).Tag, 0);
    FRadioButtonResult := (Sender as TRadioButton).Tag;
  end
  else
  if Sender is TButton then
  begin
    if SendNotify(TDN_BUTTON_CLICKED, (Sender as TButton).ModalResult, 0) <> 0 then
    begin
      Form.ModalResult := 0;
      Exit;
    end;
    Form.ModalResult := (Sender as TButton).ModalResult;
  end;
end;

procedure TTaskDialogEmu.LinkClick(Sender: TObject; const Link: string; LinkType: TSysLinkType);
begin
  SendNotify(TDN_HYPERLINK_CLICKED, 0, Cardinal(PChar(Link)));
end;

procedure TTaskDialogEmu.KeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if Key = VK_F1 then
    SendNotify(TDN_HELP);
  if (Key = VK_ESCAPE) and CanClose then
    Form.ModalResult := mrCancel;
end;

procedure TTaskDialogEmu.CloseQuery(Sender: TObject; var ACanClose: Boolean);
begin
  ACanClose := (Form.ModalResult <> mrCancel) or CanClose;
end;

procedure TTaskDialogEmu.TDMClickMessage(var Msg: TMessage);
var
  X: Integer;
begin
  for X := 0 to PanelWorkArea.ControlCount - 1 do
    if (PanelWorkArea.Controls[X] is TButton) and (TButton(PanelWorkArea.Controls[X]).ModalResult = Msg.wParam) then
      TButton(PanelWorkArea.Controls[X]).Click;
  for X := 0 to PanelButtons.ControlCount - 1 do
    if (PanelButtons.Controls[X] is TButton) and (TButton(PanelButtons.Controls[X]).ModalResult = Msg.wParam) then
      TButton(PanelButtons.Controls[X]).Click;
end;

procedure TTaskDialogEmu.TDMClickRadioButton(var Msg: TMessage);
var
  X: Integer;
begin
  FRadioBtnFirst := nil;
  for X := 0 to PanelWorkArea.ControlCount - 1 do
    if (PanelWorkArea.Controls[X] is TRadioButton) and (TRadioButton(PanelWorkArea.Controls[X]).Tag = Msg.wParam) then
      TRadioButton(PanelWorkArea.Controls[X]).Checked := True;
end;

procedure TTaskDialogEmu.TDMClickVerification(var Msg: TMessage);
begin
  if CheckBox = nil then
    Exit;
  CheckBox.Checked := Msg.wParam <> 0;
  if Msg.LParam <> 0 then
    CheckBox.SetFocus;
end;

procedure TTaskDialogEmu.TDMEnableButton(var Msg: TMessage);
var
  X: Integer;
begin
  for X := 0 to PanelWorkArea.ControlCount - 1 do
    if (PanelWorkArea.Controls[X] is TButton) and (TButton(PanelWorkArea.Controls[X]).ModalResult = Msg.wParam) then
      TButton(PanelWorkArea.Controls[X]).Enabled := Msg.lParam <> 0;
  for X := 0 to PanelButtons.ControlCount - 1 do
    if (PanelButtons.Controls[X] is TButton) and (TButton(PanelButtons.Controls[X]).ModalResult = Msg.wParam) then
      TButton(PanelButtons.Controls[X]).Enabled := Msg.lParam <> 0;
end;

procedure TTaskDialogEmu.TDMEnableRadioButton(var Msg: TMessage);
var
  X: Integer;
begin
  for X := 0 to PanelWorkArea.ControlCount - 1 do
    if (PanelWorkArea.Controls[X] is TRadioButton) and (TRadioButton(PanelWorkArea.Controls[X]).Tag = Msg.wParam) then
      TRadioButton(PanelWorkArea.Controls[X]).Enabled := Msg.lParam <> 0;
end;

procedure TTaskDialogEmu.TDMNavigatePage(var Msg: TMessage);
begin
  // TODO
end;

procedure TTaskDialogEmu.TDMSetButtonEvevation(var Msg: TMessage);
var
  X: Integer;
begin
  if Win32MajorVersion < 6 then
    Exit;
  for X := 0 to PanelWorkArea.ControlCount - 1 do
    if (PanelWorkArea.Controls[X] is TButton) and (TButton(PanelWorkArea.Controls[X]).ModalResult = Msg.wParam) then
      TButton(PanelWorkArea.Controls[X]).ElevationRequired := Msg.lParam <> 0;
  for X := 0 to PanelButtons.ControlCount - 1 do
    if (PanelButtons.Controls[X] is TButton) and (TButton(PanelButtons.Controls[X]).ModalResult = Msg.wParam) then
      TButton(PanelButtons.Controls[X]).ElevationRequired := Msg.lParam <> 0;
end;

procedure TTaskDialogEmu.TDMSetElementText(var Msg: TMessage);

  function SetCaptionResize(APanel: TPanel; const ALabel: TLabel; const ACaption: String): Integer; overload;
  begin
    Result := 0;
    if ALabel.Caption = ACaption then
      Exit;

    ALabel.Tag := ALabel.Height;
    ALabel.Caption := ACaption;
    SetLabelHeight(ALabel);

    Result := ALabel.Height - ALabel.Tag;
    if Result = 0 then
      Exit;

    Form.Height := Form.Height + Result;

    if APanel = PanelExpand then
    begin
      APanel.Height := APanel.Height + Result;
      if APanel.Parent = PanelButtons then
        APanel := PanelButtons;
    end;
    if APanel = PanelButtons then
      APanel.Height := APanel.Height + Result;
  end;

  function SetCaptionResize(const APanel: TPanel; const ALabel: TLinkLabel; const ACaption: String): Integer; overload;
  begin
    Result := 0;
    if ALabel.Caption = ACaption then
      Exit;

    ALabel.Tag := ALabel.Height;
    ALabel.Caption := ACaption;
    SetLabelHeight(ALabel);

    Result := ALabel.Height - ALabel.Tag;
    if Result = 0 then
      Exit;

    Form.Height := Form.Height + Result;

    if APanel = PanelButtons then
      APanel.Height := APanel.Height + Result;
    if APanel = PanelExpand then
      APanel.Height := APanel.Height + Result;
  end;

  procedure MoveControls(const APanel: TPanel; const ALabelTop, AMoveDelta: Integer);
  var
    X: Integer;
  begin
    for X := 0 to APanel.ControlCount - 1 do
      if APanel.Controls[X].Top > ALabelTop then
        APanel.Controls[X].Top := APanel.Controls[X].Top + AMoveDelta;
  end;

  procedure SetCaptionResizeAndMove(const APanel: TPanel; const ALabel: TLabel; const ACaption: String); overload;
  var
    MoveDelta: Integer;
  begin
    MoveDelta := SetCaptionResize(APanel, ALabel, ACaption);
    if MoveDelta = 0 then
      Exit;
    MoveControls(APanel, ALabel.Top, MoveDelta);
  end;

  procedure SetCaptionResizeAndMove(const APanel: TPanel; const ALabel: TLinkLabel; const ACaption: String); overload;
  var
    MoveDelta: Integer;
  begin
    MoveDelta := SetCaptionResize(APanel, ALabel, ACaption);
    if MoveDelta = 0 then
      Exit;
    MoveControls(APanel, ALabel.Top, MoveDelta);
  end;

var
  S: String;
begin
  BeginUpdate;
  try
    S := LoadText(PWideChar(Msg.lParam));
    case Msg.wParam of
      TDE_CONTENT:
        if Assigned(Text1) then
          SetCaptionResizeAndMove(PanelWorkArea, Text1, S)
        else
          SetCaptionResizeAndMove(PanelWorkArea, Text2, S);
      TDE_MAIN_INSTRUCTION:
        SetCaptionResizeAndMove(PanelWorkArea, Title, S);
      TDE_EXPANDED_INFORMATION:
        if DetailedText <> nil then
          SetCaptionResize(PanelExpand, DetailedText, S);
      TDE_FOOTER:
        if FooterText1 <> nil then
          SetCaptionResize(PanelButtons, FooterText1, S)
        else
        if FooterText2 <> nil then
          SetCaptionResize(PanelButtons, FooterText2, S);
    end;
  finally
    EndUpdate;
  end;
end;

procedure TTaskDialogEmu.TDMSetMarqueeProgressBar(var Msg: TMessage);
begin
  if not Assigned(ProgressBar) then
    Exit;

  if Msg.wParam = 0 then
    ProgressBar.Style := pbstNormal
  else
    ProgressBar.Style := pbstMarquee;
end;

procedure TTaskDialogEmu.TDMSetProgressBarMarquee(var Msg: TMessage);
begin
  if not Assigned(ProgressBar) then
    Exit;

  if Msg.wParam = 0 then
    ProgressBar.Style := pbstNormal
  else
    ProgressBar.Style := pbstMarquee;
  ProgressBar.MarqueeInterval := Msg.lParam;
end;

procedure TTaskDialogEmu.TDMSetProgressBarPos(var Msg: TMessage);
begin
  if not Assigned(ProgressBar) then
    Exit;

  Msg.Result := ProgressBar.Position;
  ProgressBar.Position := Msg.wParam;
end;

procedure TTaskDialogEmu.TDMSetProgressBarRange(var Msg: TMessage);
begin
  if not Assigned(ProgressBar) then
    Exit;

  Msg.Result := MakeLParam(ProgressBar.Min, ProgressBar.Max);
  ProgressBar.Min := Msg.lParam and $FFFF;
  ProgressBar.Max := (Msg.lParam shr 16) and $FFFF;
end;

procedure TTaskDialogEmu.TDMSetProgressBarState(var Msg: TMessage);
begin
  if not Assigned(ProgressBar) then
    Exit;

  Msg.Result := 0;
  case Msg.wParam of
    PBST_NORMAL:
      ProgressBar.State := pbsNormal;
    PBST_ERROR:
      ProgressBar.State := pbsError;
    PBST_PAUSED:
      ProgressBar.State := pbsPaused;
  end;
end;

procedure TTaskDialogEmu.TDMUpdateElementText(var Msg: TMessage);
var
  S: String;
begin
  S := LoadText(PWideChar(Msg.lParam));
  case Msg.wParam of
    TDE_CONTENT:
      if Assigned(Text1) then
        Text1.Caption := S
      else
        Text2.Caption := S;
    TDE_MAIN_INSTRUCTION:
      Title.Caption := S;
    TDE_EXPANDED_INFORMATION:
      if DetailedText <> nil then
        DetailedText.Caption := S;
    TDE_FOOTER:
      if FooterText1 <> nil then
        FooterText1.Caption := S
      else
      if FooterText2 <> nil then
        FooterText2.Caption := S;
  end;
end;

procedure TTaskDialogEmu.TDMUpdateIcon(var Msg: TMessage);
begin
  case Msg.wParam of
    TDIE_ICON_MAIN:
    begin
      Icon.Visible := Msg.lParam <> 0;
      SetIcon(Icon.Picture.Icon, Msg.lParam, PChar(Msg.lParam), (TaskConfig.dwFlags and TDF_USE_HICON_MAIN) <> 0);
    end;
    TDIE_ICON_FOOTER:
    begin
      FooterIcon.Visible := Msg.lParam <> 0;
      SetIcon(FooterIcon.Picture.Icon, Msg.lParam, PChar(Msg.lParam), (TaskConfig.dwFlags and TDF_USE_HICON_FOOTER) <> 0, 24);
    end;
  end;
end;

procedure TTaskDialogEmu.Tick(Sender: TObject);
begin
  if SendNotify(TDN_TIMER, GetTickCount - Cardinal(Timer.Tag), 0) <> 0 then
    Timer.Tag := GetTickCount;
end;

function TTaskDialogEmu.SendNotify(const AMsg, wParam,
  lParam: Cardinal): Cardinal;
begin
  if Assigned(TaskConfig.pfCallback) then
    Result := TaskConfig.pfCallback(Form.Handle, AMsg, wParam, lParam, TaskConfig.lpCallbackData)
  else
    Result := 0;
end;

function TTaskDialogEmu.LoadText(const AText: PWideChar): String;
begin
  if (AText = nil) or (Cardinal(AText) > SizeOf(Word)) then
    Result := AText
  else
  begin
    SetLength(Result, 4096);
    SetLength(Result, LoadString(TaskConfig.hInstance, Cardinal(AText), PChar(Result), Length(Result) + 1));
  end;
end;

function TTaskDialogEmu.TextSize(const AFont: TFont; const AText: String; const AWidth: Integer): Integer;
var
  Bmp: TBitmap;
  Flags: Cardinal;
  DC: HDC;
  OldMode, OldFont: Cardinal;
  Rec: TRect;
begin
  Bmp := TBitmap.Create;
  try
    Bmp.Width := 1;
    Bmp.Height := 1;

    DC := Bmp.Canvas.Handle;

    OldFont := SelectObject(DC, AFont.Handle);
    try
      OldMode := GetTextAlign(DC);
      try
        SetTextAlign(DC, TA_NOUPDATECP or TA_LEFT or TA_TOP);

        Flags := DT_CALCRECT or DT_WORDBREAK or DT_LEFT or DT_TOP or DT_NOPREFIX;
        if AWidth = 0 then
          Rec := Rect(0, 0, 10240, 5)
        else
          Rec := Rect(0, 0, AWidth, 5);
        Result := DrawText(DC, PChar(AText), Length(AText), Rec, Flags);
        if AWidth = 0 then
          Result := Rec.Right;
      finally
        SetTextAlign(DC, OldMode);
      end;
    finally
      SelectObject(DC, OldFont);
    end;
  finally
    FreeAndNil(Bmp);
  end;
end;

procedure TTaskDialogEmu.SetIcon(const AIcon: TIcon; const AIconH: HICON; const AIconN: PWideChar; const AUseHandle: Boolean; const AOverrideWidth: Integer);
var
  IconID: PChar;
  Ico: HICON;
  LoadIconWithScaleDown: function(hinst: HMODULE; pszName: PWideChar; cx, cy: Integer; out Ico: HICON): HRESULT; stdcall;
  IconWidth: Integer;
  IconHeight: Integer;
  Bmp: TBitmap;
  SizedBmp: TBitmap;
begin
  if AUseHandle then
    AIcon.Handle := CopyIcon(AIconH)
  else
  begin
    if TaskConfig.hInstance = 0 then
    begin
      if AIconN = TD_WARNING_ICON then
        IconID := IDI_WARNING
      else
      if AIconN = TD_ERROR_ICON then
        IconID := IDI_ERROR
      else
      if AIconN = TD_INFORMATION_ICON then
        IconID := IDI_INFORMATION
      else
      if AIconN = TD_SHIELD_ICON then
        IconID := IDI_SHIELD
      else
        IconID := nil;
    end
    else
      IconID := AIconN;

    if (IconID = IDI_SHIELD) and (Win32MajorVersion < 6) then
      IconID := IDI_WARNING;

    AIcon.Handle := 0;
    LoadIconWithScaleDown := GetProcAddress(ComCtl32DLL, 'LoadIconWithScaleDown');
    if AOverrideWidth <> 0 then
    begin
      IconWidth := AOverrideWidth;
      IconHeight := AOverrideWidth;
    end
    else
    begin
      IconWidth := GetSystemMetrics(SM_CXICON);
      IconHeight := GetSystemMetrics(SM_CYICON);
    end;

    if Assigned(LoadIconWithScaleDown) then
    begin
      if Succeeded(LoadIconWithScaleDown(TaskConfig.hInstance, IconID, IconWidth, IconHeight, Ico)) then
        AIcon.Handle := Ico;
    end;
    if AIcon.Handle = 0 then
      AIcon.Handle := LoadImageW(TaskConfig.hInstance, IconID, IMAGE_ICON, IconWidth, IconHeight, 0);
    if AIcon.Handle = 0 then
      AIcon.Handle := CopyIcon(LoadIconW(TaskConfig.hInstance, IconID));
  end;

  if AIcon.Handle = 0 then
    RaiseLastOSError;

  if (AOverrideWidth <> 0) and (AIcon.Width <> AOverrideWidth) then
  begin
    Bmp := TBitmap.Create;
    try
      Bmp.SetSize(AIcon.Width, AIcon.Height);
      Bmp.Canvas.Draw(0, 0, AIcon);
      SizedBmp := TBitmap.Create;
      try
        SizedBmp.SetSize(AOverrideWidth, AOverrideWidth);
        SizedBmp.Canvas.StretchDraw(Rect(0, 0, SizedBmp.Width, SizedBmp.Height), Bmp);
        with TImageList.CreateSize(SizedBmp.Width, SizedBmp.Height) do
        try
          AllocBy := 1;
          AddMasked(SizedBmp, BkColor);
          GetIcon(0, AIcon);
        finally
          Free;
        end;
      finally
        FreeAndNil(SizedBmp);
      end;
    finally
      FreeAndNil(Bmp);
    end;
  end;
end;

procedure TTaskDialogEmu.SetLabelHeight(const ALabel: TLabel);
var
  Rec: TRect;
begin
  Rec := ALabel.BoundsRect;

  ALabel.Height := TextSize(ALabel.Font, ALabel.Caption, Rec.Right - Rec.Left);
end;

procedure TTaskDialogEmu.SetLabelHeight(const ALabel: TLinkLabel);
var
  Rec: TRect;
begin
  Rec := ALabel.BoundsRect;

  ALabel.Height := TextSize(ALabel.Font, ALabel.Caption, Rec.Right - Rec.Left);
end;

{ TTaskForm }

constructor TTaskForm.CreateNew(AOwner: TComponent; Dummy: Integer);
begin
  FTaskDialog := TTaskDialogEmu(Dummy);
  FRTL := (TaskDialog.TaskConfig.dwFlags and TDF_RTL_LAYOUT) <> 0;
  inherited CreateNew(nil, 0);
end;

procedure TTaskForm.CreateParams(var Params: TCreateParams);
begin
  inherited;
  if FRTL then
    Params.ExStyle := Params.ExStyle or WS_EX_LAYOUTRTL;
end;

procedure TTaskForm.TDMClickMessage(var Msg: TMessage);
begin
  FTaskDialog.TDMClickMessage(Msg);
end;

procedure TTaskForm.TDMClickRadioButton(var Msg: TMessage);
begin
  FTaskDialog.TDMClickRadioButton(Msg);
end;

procedure TTaskForm.TDMClickVerification(var Msg: TMessage);
begin
  FTaskDialog.TDMClickVerification(Msg);
end;

procedure TTaskForm.TDMEnableButton(var Msg: TMessage);
begin
  FTaskDialog.TDMEnableButton(Msg);
end;

procedure TTaskForm.TDMEnableRadioButton(var Msg: TMessage);
begin
  FTaskDialog.TDMEnableRadioButton(Msg);
end;

procedure TTaskForm.TDMNavigatePage(var Msg: TMessage);
begin
  FTaskDialog.TDMNavigatePage(Msg);
end;

procedure TTaskForm.TDMSetButtonEvevation(var Msg: TMessage);
begin
  FTaskDialog.TDMSetButtonEvevation(Msg);
end;

procedure TTaskForm.TDMSetElementText(var Msg: TMessage);
begin
  FTaskDialog.TDMSetElementText(Msg);
end;

procedure TTaskForm.TDMSetMarqueeProgressBar(var Msg: TMessage);
begin
  FTaskDialog.TDMSetMarqueeProgressBar(Msg);
end;

procedure TTaskForm.TDMSetProgressBarMarquee(var Msg: TMessage);
begin
  FTaskDialog.TDMSetProgressBarMarquee(Msg);
end;

procedure TTaskForm.TDMSetProgressBarPos(var Msg: TMessage);
begin
  FTaskDialog.TDMSetProgressBarPos(Msg);
end;

procedure TTaskForm.TDMSetProgressBarRange(var Msg: TMessage);
begin
  FTaskDialog.TDMSetProgressBarRange(Msg);
end;

procedure TTaskForm.TDMSetProgressBarState(var Msg: TMessage);
begin
  FTaskDialog.TDMSetProgressBarState(Msg);
end;

procedure TTaskForm.TDMUpdateElementText(var Msg: TMessage);
begin
  FTaskDialog.TDMUpdateElementText(Msg);
end;

procedure TTaskForm.TDMUpdateIcon(var Msg: TMessage);
begin
  FTaskDialog.TDMUpdateIcon(Msg);
end;

end.
